<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
use Elementor\Controls_Manager;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;

/**
 * Elementor heading widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class GVAElement_Image_Content_Parallax extends GVAElement_Base {    

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'gva-image-content-parallax';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'GVA Image Content Parallax', 'gaviasthemer' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-type-tool';
	}


	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'image', 'content', 'parallax' ];
	}

	public function get_script_depends() {
		return [
         //'gva-skrollr-js'
     ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => __( 'Content & Images', 'gaviasthemer' ),
			]
		);
		$this->add_control(
			'sub_title',
			[
				'label' => __( 'Sub Title', 'gaviasthemer' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your Sub Title', 'gaviasthemer' ),
				'label_block' => true
			]
		);
		$this->add_control(
			'title_text',
			[
				'label' => __( 'Title', 'gaviasthemer' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your title', 'gaviasthemer' ),
				'default' => __( 'Add Your Heading Text Here', 'gaviasthemer' ),
				'label_block' => true
			]
		);
		$this->add_control(
			'description_text',
			[
				'label' => __( 'Description', 'gaviasthemer' ),
				'type' => Controls_Manager::TEXTAREA,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'Enter Your Description', 'gaviasthemer' ),
			]
		);
		$this->add_control(
			'image',
			[
				'label' => __( 'Choose Image', 'gaviasthemer' ),
				'type' => Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);
		$this->add_group_control(
         Elementor\Group_Control_Image_Size::get_type(),
         [
             'name'      => 'image',
             'default'   => 'full',
             'separator' => 'none',
         ]
      );

		$this->add_control(
			'image-2',
			[
				'label' => __( 'Choose Image II (Small Image)', 'gaviasthemer' ),
				'type' => Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);
		$this->add_control(
			'position',
			[
				'label' => __( 'Content Position', 'gaviasthemer' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'left' => esc_html__('Left', 'gaviasthemer'),
					'right' => esc_html__('Right', 'gaviasthemer'),
				],
				'default' => 'left',
			]
		);
		$this->add_control(
			'style',
			[
				'label' => __( 'Style', 'gaviasthemer' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'style-v1' => esc_html__('Style I', 'gaviasthemer'),
					'style-v2' => esc_html__('Style II', 'gaviasthemer'),
				],
				'default' => 'style-v1',
			]
		);
		$this->add_control(
			'header_tag',
			[
				'label' => __( 'HTML Tag', 'gaviasthemer' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h2',
			]
		);
		$this->add_control(
			'link',
			[
				'label' => __( 'Link', 'gaviasthemer' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'https://your-link.com', 'gaviasthemer' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'link_text',
			[
				'label' => __( 'Text Link', 'gaviasthemer' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'Read More', 'gaviasthemer' ),
				'default' => __( 'Read More', 'gaviasthemer' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => __( 'Title', 'gaviasthemer' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Text Color', 'gaviasthemer' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					// Stronger selector to avoid section style from overwriting
					'{{WRAPPER}}.elementor-widget-heading .elementor-heading-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'typography',
				'selector' => '{{WRAPPER}} .elementor-heading-title',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['title_text'] ) ) {
			return;
		}
		$style = $settings['style'];
		$position = $settings['position'];
		$sub_title = $settings['sub_title'];
		$title_text = $settings['title_text'];
		$description_text = $settings['description_text'];
		$header_tag = 'h2';
		if(!empty($settings['header_tag'])) $header_tag = $settings['header_tag'];
		
		$this->add_render_attribute( 'block', 'class', [ 'gsc-content-images-parallax', $settings['style'], $settings['position'] ] );
		$this->add_render_attribute( 'title_text', 'class', 'title' );
		$this->add_render_attribute( 'description_text', 'class', 'description' );

		$this->add_inline_editing_attributes( 'title_text', 'none' );
		$this->add_inline_editing_attributes( 'description_text' );

		$title_html = $settings['title_text'];
       if (!empty($settings['link']['url'])) :
           $team_name_html = '<a href="' . esc_url($settings['link']['url']) . '">' . $title_html . '</a>';
       endif;

      $skrollable1 = '';
      $skrollable2 = '';
      if($style == 'style-v1'){
        if($position == 'left') $skrollable2 = 'data-bottom-top="bottom: 180px;" data-top-bottom="bottom: -30px;"';
        if($position == 'right') $skrollable2 = 'data-bottom-top="top: 180px;" data-top-bottom="top: -30px;"';
      }
      if($style == 'style-v2'){
        if($position == 'left'){
          $skrollable1 = 'data-top="top: 40px;" data-bottom="top: 0px;"';
          $skrollable2 = 'data-top="top: 0px;" data-bottom="top: 80px;"';
        }
        if($position == 'right'){
          $skrollable1 = 'data-top="top: 0px;" data-bottom="top: 80px;"';
          $skrollable2 = 'data-top="top: 40px;" data-bottom="top: 0px;"';
        }
      }

		?>
		
		   <div <?php echo $this->get_render_attribute_string( 'block' ) ?>>
           <div class="widget-content">
             <div class="images">

             	<?php if (!empty($settings['image']['url'])) : ?>
			         <div class="image-1" <?php echo $skrollable1 ?>>
		            	<?php
	                    $image_html = Group_Control_Image_Size::get_attachment_image_html($settings, 'image');
	                    echo $image_html;
		               ?>
			         </div>
		         <?php endif; ?>
		         <?php if (!empty($settings['image-2']['url'])) : ?>
			         <div class="image-2" <?php echo $skrollable2 ?>>
		            	<?php
	                    $image_html = Group_Control_Image_Size::get_attachment_image_html($settings, 'image-2');
	                    echo $image_html;
		               ?>
			         </div>
		         <?php endif; ?>

             </div>
             
             <div class="content">
               <div class="content-inner">
                 	
                 	<?php if($sub_title){ ?> 
			         	<div class="sub-title"><span><?php echo esc_html($sub_title); ?></span></div>
			         <?php } ?>

                 	<?php if($title_text){ ?>
			         	<<?php echo esc_attr($header_tag) ?> <?php echo $this->get_render_attribute_string( 'title_text' ); ?>>
			         		<?php echo $title_html; ?>
			         	</<?php echo esc_attr($header_tag) ?>>
			         <?php } ?>
                 
                 	<?php if(!empty($settings['description_text'])){ ?>
                 		<div <?php echo $this->get_render_attribute_string( 'description_text' ); ?>><?php echo wp_kses($description_text, true); ?></div>
                 	<?php } ?>
                 		
                 <?php if(!empty($settings['link']['url'])){ ?>
                   <div class="action">
                     <a class="btn-theme" href="<?php echo $settings['link']['url'] ?>"><?php echo $settings['link_text'] ?></a>
                   </div>
                 <?php } ?>           
               </div>
             </div>  

           </div>
         </div>

		<?php
	}

}
